package jbugs.land;

import jbugs.devices.VirtualScreen;
import jbugs.utils.BugMath;

public class Terrain {
	
	// ------ Constants ------
	
	/**
	 * Denotes an empty spot on the 'world'
	 */
	//final public static char SPC = ' ';
	final public static char SPC = ' ';
	
	/**
	 * Denotes a mountain on the 'world'
	 */
	final public static char MNT = '#';
	
	/**
	 * Denotes water on the 'world'
	 */
	final public static char WTR = '~';
	
	/**
	 * The height of water level
	 */
	final public static int WTR_LVL = 0;
	
	/**
	 * The height of a mountain
	 */
	final public static int MNT_LVL = 100;
	
	/**
	 * Height
	 */
	final public static int HEIGHT = 100;
	
	// ------ Data members ------
	/**
	 * Screen to draw to
	 */
	private VirtualScreen screen;
	
	// ------ Constructors ------
	
	/**
	 * Creates a new terrain generator with default parameters
	 */
	public Terrain() {
		screen = null;
	}
	
	// ------ Methods ------
	
	/**
	 * Sets the virtual screen to draw from
	 * @param	new_screen	VirtualScreen to draw from
	 */
	public void setVirtualScreen(VirtualScreen new_screen) {
		screen = new_screen;
	}
	
	/**
	 * Creates the terrain for the 'world'
	 */
	public void generateTerrain(int grid_size, int roughness) {
		final int world_row = screen.getHeight(), world_col = screen.getWidth();
		int elevation_map[][] = null;
		int x,y;
		int grid_s = (int)(Math.pow(2,grid_size));
		
		elevation_map = new int[world_row][world_col];
		
		for (x = 0; x < world_col; x+= grid_s) {
			for (y = 0; y < world_row; y += grid_s) {
				elevation_map[y][x] = BugMath.getRandom(0,Terrain.HEIGHT);
			} // for y
		} // for x
		
		for (grid_s /= 2; grid_s >= 1; grid_s /= 2) {
		
			/* Interpolate smaller grid along horizontals */
			for (x = grid_s; x < world_col; x += grid_s * 2) {
				for (y = 0; y < world_row; y += grid_s * 2) {
					elevation_map[y][x] = BugMath.getRandom(-roughness * BugMath.log10(grid_s),
															 roughness * BugMath.log10(grid_s)) +
										  (elevation_map[y][x - grid_s] +
										   elevation_map[y][((x + grid_s < world_col) ? (x + grid_s) : (0))]) / 2; 
				} // for y
			} // for x
	
			/* Interpolate smaller grid along the verticals */
			for (x = 0; x < world_col; x += grid_s * 2) {
				for (y = grid_s; y < world_row; y += grid_s * 2) {
					elevation_map[y][x] = BugMath.getRandom(-roughness * BugMath.log10(grid_s),
														     roughness * BugMath.log10(grid_s)) +
										  (elevation_map[(y - grid_s)][x] +
										   elevation_map[(y + grid_s < world_row) ? (y + grid_s) : (0)][x]) / 2;
				} // for y
			} // for x
			
			
			/* Interpolate smaller grid between gridlines */
			for (x = grid_s; x < world_col; x += grid_s * 2) {
				for (y = grid_s; y < world_row; y += grid_s * 2) {
					elevation_map[y][x] = BugMath.getRandom(-roughness * BugMath.log10(grid_s),
															 roughness * BugMath.log10(grid_s)) +
										  (elevation_map[(y - grid_s)][x] +
										   elevation_map[(y + grid_s < world_row) ? (y + grid_s) : (0)][x] +
										   elevation_map[y][(x+grid_s < world_col) ? (x + grid_s) : (0)] +
										   elevation_map[y][(x - grid_s)]) / 4;
				} // for y
			} // for x
		} // for grid_s
		
		/* Now, place the WTR (water) & MNT (mountains) */
		for (x = 0; x < world_col; x++) {
			for (y = 0; y < world_row; y++) {
				if (elevation_map[y][x] <= Terrain.WTR_LVL)
					screen.setCharAt(y,x,Terrain.WTR);
				else if (elevation_map[y][x] >= Terrain.MNT_LVL)
					screen.setCharAt(y,x,Terrain.MNT);
				else
					screen.setCharAt(y,x,Terrain.SPC);
			} // for y
		} // for x
		
		// Perform some clean up & run the garbage collector
		elevation_map = null;
		System.gc();
	}	
}
